<?php

$trades_file = 'trades.json';
$trade_id = filter_input(INPUT_GET, 'id'); // Get ID from URL parameter
$trade = null;
$error_message = '';

// Function to read trades (can be simplified if only used here)
function findTradeById(string $filename, $id): ?array {
    if (empty($id) || !file_exists($filename)) {
        return null;
    }
    $json_data = @file_get_contents($filename);
    if ($json_data === false) return null;
    $trades = json_decode($json_data, true);
     if (!is_array($trades)) return null;

    foreach ($trades as $t) {
        // Use loose comparison (==) in case ID types differ (e.g., string vs int)
        if (isset($t['id']) && $t['id'] == $id) {
            return $t;
        }
    }
    return null; // Not found
}

// --- Find the trade ---
if (empty($trade_id)) {
    $error_message = "No Trade ID provided.";
} else {
    $trade = findTradeById($trades_file, $trade_id);
    if ($trade === null) {
        $error_message = "Trade with ID '" . htmlspecialchars($trade_id) . "' not found.";
    }
}

// --- Helper for formatting dates (optional) ---
function formatDisplayDate(?string $isoDate): string {
    if (empty($isoDate)) return 'N/A';
    try {
        // Example: Format as "Apr 4, 2025, 10:30 AM" (Adjust format as needed)
        return (new DateTime($isoDate))->format('M j, Y, g:i A');
    } catch (Exception $e) {
        return 'Invalid Date';
    }
}

// --- Helper for calculating relative time (e.g., "40 minutes ago") ---
function timeAgo(string $datetime, string $full = null ): string {
    if(empty($datetime)) return 'N/A';
    try{
        $now = new DateTime;
        $ago = new DateTime($datetime);
        $diff = $now->diff($ago);

        $diff->w = floor($diff->d / 7);
        $diff->d -= $diff->w * 7;

        $string = array(
            'y' => 'year', 'm' => 'month', 'w' => 'week', 'd' => 'day',
            'h' => 'hour', 'i' => 'minute', 's' => 'second',
        );
        foreach ($string as $k => &$v) {
            if ($diff->$k) {
                $v = $diff->$k . ' ' . $v . ($diff->$k > 1 ? 's' : '');
            } else {
                unset($string[$k]);
            }
        }

        if (!$full) $string = array_slice($string, 0, 1);
        return $string ? implode(', ', $string) . ' ago' : 'just now';

    } catch (Exception $e) {
        return 'N/A'; // Handle invalid date format gracefully
    }
}


?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Trade Details</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" integrity="sha512-9usAa10IRO0HhonpyAIVpjrylPvoDwiPUiKdWk5t3PyolY1cOd4DSE0Ga+ri4AuTroPR5aQvXU9xC6qOPnzFeg==" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <style>
        /* Reusing and adapting styles from the front-end card design */
        :root {
            --bg-main: #f8f9fa;
            --bg-card: #ffffff;
            --text-primary: #212529;
            --text-secondary: #6c757d;
            --border-color: #dee2e6;
            --highlight-green: #16c784; /* Green */
            --highlight-red: #ea3943;   /* Red */
            --status-closed-color: #6c757d; /* Grey */
            --status-open-color: var(--highlight-green);
            --leverage-bg: rgba(0, 0, 0, 0.05);
            /* Add icon colors if needed, e.g. var(--icon-btc): #f7931a; */
        }
         body { margin: 0; padding: 20px; font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif; background-color: var(--bg-main); color: var(--text-primary); line-height: 1.6; }
        .container { max-width: 700px; margin: 20px auto; }
        .trade-card {
            background-color: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 10px;
            overflow: hidden;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
            margin-bottom: 20px;
        }
        .trade-card.status-closed { border-left: 5px solid var(--status-closed-color); }
        .trade-card.status-open { border-left: 5px solid var(--status-open-color); }

        .card-header { display: flex; justify-content: space-between; align-items: center; padding: 15px 20px; background-color: rgba(0,0,0,0.03); border-bottom: 1px solid var(--border-color); }
        .pair-info { display: flex; align-items: center; gap: 12px; }
        .pair-info i { font-size: 1.8em; /* Adjust as needed based on icon */ color: #495057; }
        /* Add specific icon colors here if desired */
        .pair-info .btc-icon { color: #f7931a; }
        .pair-info .eth-icon { color: #627eea; }
        .pair-info .xau-icon { color: #d4af37; } /* Gold color */
        /* Add more specific icon colors */

        .pair-info .pair-name { font-size: 1.2em; font-weight: 500; color: var(--text-primary); }
        .pair-info .leverage { font-size: 0.85em; font-weight: 500; padding: 3px 8px; border-radius: 4px; background-color: var(--leverage-bg); margin-left: 5px; color: var(--text-secondary); }
        .pair-info .leverage.buy { color: var(--highlight-green); border: 1px solid currentColor; background: transparent;}
        .pair-info .leverage.sell { color: var(--highlight-red); border: 1px solid currentColor; background: transparent;}


        .trade-status { display: flex; align-items: center; gap: 8px; font-size: 0.9em; padding: 5px 12px; border-radius: 15px; font-weight: 500; color: #fff; }
        .trade-card.status-closed .trade-status { background-color: var(--status-closed-color); }
        .trade-card.status-open .trade-status { background-color: var(--status-open-color); }

        .card-body { padding: 25px; }
        .roi-section { text-align: center; margin-bottom: 25px; padding-bottom: 20px; border-bottom: 1px solid var(--border-color); }
        .roi-section .roi-label { display: block; font-size: 0.9em; color: var(--text-secondary); margin-bottom: 5px; text-transform: uppercase; letter-spacing: 0.5px; }
        .roi-section .roi-value { display: block; font-size: 2.4em; font-weight: 700; line-height: 1.2; }
        .roi-value.profit { color: var(--highlight-green); }
        .roi-value.loss { color: var(--highlight-red); }

        .details-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 18px 25px; }
        .detail-item { display: flex; flex-direction: column; }
        .detail-item .label { color: var(--text-secondary); font-size: 0.85em; margin-bottom: 5px; display: flex; align-items: center; gap: 7px; text-transform: uppercase; letter-spacing: 0.3px; }
        .detail-item .label i { width: 15px; text-align: center; color: var(--text-secondary); opacity: 0.8; }
        .detail-item .value { font-weight: 500; font-size: 1.05em; color: var(--text-primary); }
        .detail-item .value.profit { color: var(--highlight-green); }
        .detail-item .value.loss { color: var(--highlight-red); }

        .card-footer { background-color: rgba(0,0,0,0.02); padding: 10px 20px; text-align: right; font-size: 0.8em; color: var(--text-secondary); border-top: 1px solid var(--border-color); }
        .card-footer i { margin-right: 5px; }

        .message { border: 1px solid transparent; padding: 15px; margin: 20px auto; border-radius: 4px; text-align: center; font-weight: 500; max-width: 650px; }
        .error-message { background-color: #f8d7da; color: #721c24; border-color: #f5c6cb; }
        .info-message { background-color: #e2e3e5; color: #383d41; border-color: #d6d8db; }
        .back-link { display: inline-block; margin-top: 20px; padding: 8px 15px; background-color: #6c757d; color: #fff; text-decoration: none; border-radius: 4px; transition: background-color 0.2s; }
        .back-link:hover { background-color: #5a6268; }

    </style>
</head>
<body>

    <div class="container">

        <?php if ($trade): ?>
            <h1>Trade Added Successfully!</h1>
            <article class="trade-card status-<?php echo strtolower(htmlspecialchars($trade['status'])); ?>">
                <header class="card-header">
                    <div class="pair-info">
                        <i class="<?php echo htmlspecialchars($trade['iconClass'] ?? 'fas fa-chart-line'); ?>"></i>
                        <span class="pair-name"><?php echo htmlspecialchars($trade['pair']); ?></span>
                        <?php if (!empty($trade['leverage'])):
                            // Basic check for leverage direction for coloring
                            $leverageClass = '';
                            if (stripos($trade['leverage'], 'long') !== false) $leverageClass = 'buy';
                            elseif (stripos($trade['leverage'], 'short') !== false) $leverageClass = 'sell';
                        ?>
                            <span class="leverage <?php echo $leverageClass; ?>"><?php echo htmlspecialchars($trade['leverage']); ?></span>
                        <?php endif; ?>
                    </div>
                    <div class="trade-status">
                        <span><?php echo htmlspecialchars($trade['status']); ?></span>
                        <i class="fas <?php echo ($trade['status'] === 'Closed') ? 'fa-check-circle' : 'fa-hourglass-half'; ?>"></i>
                    </div>
                </header>

                 <section class="card-body">
                    <?php if (isset($trade['roi'])): ?>
                    <div class="roi-section">
                        <span class="roi-label">ROI</span>
                        <span class="roi-value <?php echo ($trade['roi'] >= 0) ? 'profit' : 'loss'; ?>">
                            <?php echo (($trade['roi'] >= 0) ? '+' : '') . number_format($trade['roi'], 2); ?>%
                        </span>
                    </div>
                    <?php endif; ?>

                    <div class="details-grid">
                        <div class="detail-item">
                            <span class="label"><i class="fas fa-sign-in-alt"></i> Entry Price</span>
                            <span class="value"><?php echo number_format($trade['entryPrice'] ?? 0, 2); ?></span>
                        </div>
                         <div class="detail-item">
                             <span class="label">
                                <i class="fas <?php echo ($trade['status'] === 'Closed') ? 'fa-sign-out-alt' : 'fa-chart-line'; ?>"></i>
                                <?php echo ($trade['status'] === 'Closed') ? 'Exit Price' : 'Current Price'; ?>
                             </span>
                            <span class="value"><?php echo isset($trade['exitPrice']) ? number_format($trade['exitPrice'], 2) : 'N/A'; ?></span>
                        </div>
                         <div class="detail-item">
                            <span class="label"><i class="fas fa-calendar-alt"></i> Opened</span>
                            <span class="value"><?php echo formatDisplayDate($trade['dateOpened']); ?></span>
                        </div>
                        <div class="detail-item">
                            <span class="label"><i class="fas fa-calendar-check"></i> Closed</span>
                            <span class="value"><?php echo formatDisplayDate($trade['dateClosed']); ?></span>
                             <?php if($trade['status'] === 'Closed'): ?>
                                <span style="font-size: 0.8em; color: var(--text-secondary); margin-top: 2px;">(<?php echo timeAgo($trade['dateClosed']); ?>)</span>
                             <?php endif; ?>
                        </div>

                         <div class="detail-item">
                             <span class="label">
                                <i class="fas fa-coins"></i>
                                <?php echo ($trade['status'] === 'Closed') ? 'P & L' : 'Unrealized P&L'; ?>
                            </span>
                            <span class="value <?php echo ($trade['pnl'] >= 0) ? 'profit' : 'loss'; ?>">
                               $<?php echo number_format($trade['pnl'] ?? 0, 2); ?> </span>
                        </div>

                         <div class="detail-item">
                            <span class="label"><i class="fas fa-info-circle"></i> Trade ID</span>
                            <span class="value" style="font-size:0.9em; color: var(--text-secondary);"><?php echo htmlspecialchars($trade['id']); ?></span>
                        </div>

                    </div>
                </section>

                <footer class="card-footer">
                     <span class="timestamp"><i class="fas fa-clock"></i> Last Updated: <?php echo timeAgo($trade['lastUpdated']); ?> (<?php echo formatDisplayDate($trade['lastUpdated']); ?>)</span>
                </footer>

            </article>

             <a href="admin.php" class="back-link"><i class="fas fa-plus"></i> Add Another Trade</a>
             <?php else: ?>
            <h1>Error</h1>
            <p class="message error-message"><?php echo htmlspecialchars($error_message); ?></p>
            <a href="admin.php" class="back-link">Back to Admin Form</a>
        <?php endif; ?>

    </div>

</body>
</html>